% Copyright 2014 Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000 with Sandia
% Corporation, the U.S. Government retains certain rights in this software
%
% This file is part of Sandia SPT (Sandia Simple Particle Tracking) v. 1.0.
% 
% Sandia SPT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Sandia SPT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Sandia SPT.  If not, see <http://www.gnu.org/licenses/>.

function [pairs,cost,v,u,costMat] = callLapjv(costMat,resolution)

%function [pairs,cost,v,u,costMat] = callLapjv(costMat,resolution)
%
%PURPOSE:
%	callLapjv.m is a wrapper function to call the lapjv code available at 
%	"http://www.mathworks.com/matlabcentral/fileexchange/26836-lapjv-jonker
%	-volgenant-algorithm-for-linear-assignment-problem-v3-0"
%	The wrapper has two purposes. 
%		1) Rather than outputing the best column for each row, it outputs 
%			the selected row-column pairs. 
%		2) The wrapper compensates for a documented behavior of Version 3.0
%			of lapjv.m
%				"For a rectangular (nonsquare) costMat, rowsol is the 
%				index vector of the larger dimension assigned to the 
%				smaller dimension."
%			The wrapper ensures consistent output format regardless of
%			which dimension is larger. 
%	With the exception of the output variable pairs, all inputs and outputs
%	match those of lapjv.m
%
%DEPENDENCIES:
%	lapjv.m
%
%INPUTS:
%	costMat:
%		The cost matrix. Entries which should not be considered as possible
%		solutions should be replaced with NaN's. From lapjv.m, "the (i,j)th
%		element represents the cost to assign the jth job to the ith 
%		worker.
%	resolution: (optional)
%		See lapjv.m
%
%OUTPUTS:
%	pairs:
%		The row-column pairs corresponding to the solution of the linear
%		assignment problem. When no alternative match is available, the 
%		algorithm will select entries which were NaN or Inf. 
%	Other outputs:
%		See lapjv.m
%
%REFERENCES:
%	Calls and requires lapjv.m written by Yi Cao at Cranfield University
%	"http://www.mathworks.com/matlabcentral/fileexchange/26836-lapjv-jonker
%	-volgenant-algorithm-for-linear-assignment-problem-v3-0"
%
%	Algorithm by
%	R. Jonker and A. Volgenant, "A shortest augmenting path algorithm for
%	dense and spare linear assignment problems", Computing, Vol. 38, pp.
%	325-340, 1987.
%
%LIMITATIONS:
%	-None-
%
%NOTES:
%	-None-
%
%Written by Stephen M. Anthony 08/26/2014
%Last modified on 08/26/14 by Stephen M. Anthony

%Call LAPJV, the Jonker-Volgenant Algorithm for Linear Assignment Problem.
if nargin==2
	[rowsol,cost,v,u,costMat] = lapjv(costMat,resolution);
else
	[rowsol,cost,v,u,costMat] = lapjv(costMat);
end

%Convert the output to row and column pairs
pairs = [(1:numel(rowsol)); rowsol]';

[rdim,cdim] = size(costMat);
if rdim>cdim
	%The provided lapjv code performs a transpose when there are more rows
	%than columns in the cost matrix. While the other outputs are swapped
	%back accordingly, rowsol was not. The following corrects the output of
	%pairs to be consistent, regardless of whether there were more rows or
	%columns. 
	pairs = sortrows(pairs(:,[2 1]));
end
